import type { IncomingMessage, ServerResponse } from "http";
import type { MemoryPayload, TranscriptSegment, OmiConfig } from "./types";
import { processMemory, processTranscript, processAudioBytes } from "./processor";

function parseQuery(url: string): Record<string, string> {
  const params: Record<string, string> = {};
  const queryStart = url.indexOf("?");
  if (queryStart === -1) return params;
  
  const queryString = url.slice(queryStart + 1);
  for (const pair of queryString.split("&")) {
    const [key, value] = pair.split("=");
    if (key) params[decodeURIComponent(key)] = decodeURIComponent(value || "");
  }
  return params;
}

async function readBody(req: IncomingMessage): Promise<Buffer> {
  const chunks: Buffer[] = [];
  for await (const chunk of req) {
    chunks.push(chunk as Buffer);
  }
  return Buffer.concat(chunks);
}

export function createWebhookHandlers(api: any, config: OmiConfig) {
  const logger = api.logger;

  return {
    // Memory creation webhook
    memory: async (req: IncomingMessage, res: ServerResponse) => {
      if (req.method !== "POST") {
        res.writeHead(405);
        res.end("Method not allowed");
        return;
      }

      try {
        const body = await readBody(req);
        const payload: MemoryPayload = JSON.parse(body.toString());

        if (payload.discarded) {
          logger.debug("[omimesh] Skipping discarded memory");
          res.writeHead(200);
          res.end("OK (discarded)");
          return;
        }

        if (config.storeMemories !== false) {
          processMemory(payload, config, logger);
        }

        // Optionally notify agent
        if (config.notifyAgent && api.runtime?.injectSystemEvent) {
          const msg = `[Omi] New memory: ${payload.structured.emoji} ${payload.structured.title}\n${payload.structured.overview}`;
          api.runtime.injectSystemEvent(msg);
        }

        res.writeHead(200);
        res.end("OK");
      } catch (err: any) {
        logger.error(`[omimesh] Memory webhook error: ${err.message}`);
        res.writeHead(400);
        res.end("Bad request");
      }
    },

    // Real-time transcript webhook
    transcript: async (req: IncomingMessage, res: ServerResponse) => {
      if (req.method !== "POST") {
        res.writeHead(405);
        res.end("Method not allowed");
        return;
      }

      try {
        const query = parseQuery(req.url || "");
        const sessionId = query.session_id || "unknown";

        const body = await readBody(req);
        const segments: TranscriptSegment[] = JSON.parse(body.toString());

        if (config.storeTranscripts !== false) {
          processTranscript(segments, sessionId, config, logger);
        }

        res.writeHead(200);
        res.end("OK");
      } catch (err: any) {
        logger.error(`[omimesh] Transcript webhook error: ${err.message}`);
        res.writeHead(400);
        res.end("Bad request");
      }
    },

    // Audio bytes webhook
    audio: async (req: IncomingMessage, res: ServerResponse) => {
      if (req.method !== "POST") {
        res.writeHead(405);
        res.end("Method not allowed");
        return;
      }

      if (!config.storeAudio) {
        res.writeHead(200);
        res.end("OK (audio storage disabled)");
        return;
      }

      try {
        const query = parseQuery(req.url || "");
        const sampleRate = parseInt(query.sample_rate || "16000", 10);
        const uid = query.uid || "unknown";

        const audioBuffer = await readBody(req);
        processAudioBytes(audioBuffer, sampleRate, uid, config, logger);

        res.writeHead(200);
        res.end("OK");
      } catch (err: any) {
        logger.error(`[omimesh] Audio webhook error: ${err.message}`);
        res.writeHead(400);
        res.end("Bad request");
      }
    },

    // Health check / status
    status: async (req: IncomingMessage, res: ServerResponse) => {
      res.writeHead(200, { "Content-Type": "application/json" });
      res.end(JSON.stringify({
        ok: true,
        plugin: "omi",
        version: "1.0.0",
        config: {
          storeMemories: config.storeMemories !== false,
          storeTranscripts: config.storeTranscripts !== false,
          storeAudio: config.storeAudio === true,
          notifyAgent: config.notifyAgent === true,
        },
      }));
    },
  };
}
