import * as fs from "fs";
import * as path from "path";
import type { MemoryPayload, TranscriptSegment, OmiConfig } from "./types";

function getWorkspacePath(): string {
  return process.env.OPENCLAW_WORKSPACE || path.join(process.env.HOME || "~", ".openclaw/workspace");
}

function ensureDir(dirPath: string): void {
  if (!fs.existsSync(dirPath)) {
    fs.mkdirSync(dirPath, { recursive: true });
  }
}

function getDateStr(): string {
  return new Date().toISOString().split("T")[0];
}

function getTimeStr(): string {
  return new Date().toISOString().split("T")[1].split(".")[0];
}

export function processMemory(payload: MemoryPayload, config: OmiConfig, logger: any): void {
  const workspace = getWorkspacePath();
  const memoryDir = path.join(workspace, config.memoryPath || "memory/omi");
  ensureDir(memoryDir);

  const dateStr = getDateStr();
  const filePath = path.join(memoryDir, `${dateStr}.md`);

  // Format the memory entry
  const entry = formatMemoryEntry(payload);

  // Append to daily file
  fs.appendFileSync(filePath, entry + "\n\n");
  logger.info(`[omimesh] Stored memory: ${payload.structured.title}`);
}

function formatMemoryEntry(payload: MemoryPayload): string {
  const time = new Date(payload.created_at).toLocaleTimeString("en-US", {
    hour: "2-digit",
    minute: "2-digit",
  });

  const lines: string[] = [
    `## ${payload.structured.emoji} ${payload.structured.title}`,
    `*${time} — ${payload.structured.category}*`,
    "",
    payload.structured.overview,
  ];

  // Add action items if any
  if (payload.structured.action_items?.length > 0) {
    lines.push("", "**Action Items:**");
    for (const item of payload.structured.action_items) {
      const checkbox = item.completed ? "[x]" : "[ ]";
      lines.push(`- ${checkbox} ${item.description}`);
    }
  }

  // Add transcript summary (condensed)
  if (payload.transcript_segments?.length > 0) {
    lines.push("", "<details>", "<summary>Transcript</summary>", "");
    for (const seg of payload.transcript_segments) {
      const speaker = seg.is_user ? "You" : seg.speaker;
      lines.push(`**${speaker}:** ${seg.text}`);
    }
    lines.push("", "</details>");
  }

  return lines.join("\n");
}

export function processTranscript(
  segments: TranscriptSegment[],
  sessionId: string,
  config: OmiConfig,
  logger: any
): void {
  const workspace = getWorkspacePath();
  const memoryDir = path.join(workspace, config.memoryPath || "memory/omi");
  const transcriptDir = path.join(memoryDir, "transcripts");
  ensureDir(transcriptDir);

  const dateStr = getDateStr();
  const filePath = path.join(transcriptDir, `${dateStr}-${sessionId}.jsonl`);

  // Append segments as JSONL
  for (const seg of segments) {
    fs.appendFileSync(filePath, JSON.stringify({ ...seg, timestamp: Date.now() }) + "\n");
  }

  logger.debug(`[omimesh] Stored ${segments.length} transcript segments for session ${sessionId}`);
}

export function processAudioBytes(
  audioBuffer: Buffer,
  sampleRate: number,
  uid: string,
  config: OmiConfig,
  logger: any
): void {
  const workspace = getWorkspacePath();
  const memoryDir = path.join(workspace, config.memoryPath || "memory/omi");
  const audioDir = path.join(memoryDir, "audio");
  ensureDir(audioDir);

  const dateStr = getDateStr();
  const timeStr = getTimeStr().replace(/:/g, "-");
  const filePath = path.join(audioDir, `${dateStr}-${timeStr}-${uid}.pcm`);

  fs.writeFileSync(filePath, audioBuffer);
  logger.debug(`[omimesh] Stored audio chunk: ${audioBuffer.length} bytes @ ${sampleRate}Hz`);
}
