/**
 * OmiMesh — Omi Integration Plugin for OpenClaw
 * 
 * Connect your Omi AI wearable to OpenClaw. Memories, transcripts,
 * and context flow seamlessly into your agent.
 * 
 * Endpoints:
 *   POST /omimesh/memory     - Memory creation events
 *   POST /omimesh/transcript - Real-time transcript segments
 *   POST /omimesh/audio      - Raw audio bytes (PCM16)
 *   GET  /omimesh/status     - Plugin health check
 */

import { createWebhookHandlers } from "./src/webhooks";
import type { OmiConfig } from "./src/types";

export const id = "omimesh";
export const name = "OmiMesh";

export function register(api: any) {
  const config: OmiConfig = api.config?.plugins?.entries?.omimesh?.config || {};
  const logger = api.logger;

  logger.info("[omimesh] 🔗 Initializing OmiMesh");
  logger.info(`[omimesh] Memory storage: ${config.storeMemories !== false ? "enabled" : "disabled"}`);
  logger.info(`[omimesh] Transcript storage: ${config.storeTranscripts !== false ? "enabled" : "disabled"}`);
  logger.info(`[omimesh] Audio storage: ${config.storeAudio === true ? "enabled" : "disabled"}`);
  logger.info(`[omimesh] Agent notifications: ${config.notifyAgent === true ? "enabled" : "disabled"}`);

  const handlers = createWebhookHandlers(api, config);

  // Register HTTP handlers for webhooks
  api.registerHttpHandler({
    method: "POST",
    path: "/omimesh/memory",
    handler: handlers.memory,
  });

  api.registerHttpHandler({
    method: "POST", 
    path: "/omimesh/transcript",
    handler: handlers.transcript,
  });

  api.registerHttpHandler({
    method: "POST",
    path: "/omimesh/audio",
    handler: handlers.audio,
  });

  api.registerHttpHandler({
    method: "GET",
    path: "/omimesh/status",
    handler: handlers.status,
  });

  // Register RPC method for status
  api.registerGatewayMethod("omimesh.status", ({ respond }: any) => {
    respond(true, {
      ok: true,
      plugin: "omimesh",
      version: "1.0.0",
      endpoints: {
        memory: "/omimesh/memory",
        transcript: "/omimesh/transcript",
        audio: "/omimesh/audio",
        status: "/omimesh/status",
      },
      config: {
        storeMemories: config.storeMemories !== false,
        storeTranscripts: config.storeTranscripts !== false,
        storeAudio: config.storeAudio === true,
        memoryPath: config.memoryPath || "memory/omi",
        notifyAgent: config.notifyAgent === true,
      },
    });
  });

  // Register CLI command
  api.registerCli(
    ({ program }: any) => {
      const cmd = program
        .command("omimesh")
        .description("OmiMesh — Omi wearable integration");

      cmd
        .command("status")
        .description("Show OmiMesh status")
        .action(async () => {
          console.log("🔗 OmiMesh v1.0.0");
          console.log("");
          console.log("Endpoints:");
          console.log("  POST /omimesh/memory     - Memory creation webhook");
          console.log("  POST /omimesh/transcript - Real-time transcript webhook");
          console.log("  POST /omimesh/audio      - Audio bytes webhook");
          console.log("  GET  /omimesh/status     - Health check");
          console.log("");
          console.log("Configuration:");
          console.log(`  Store memories:    ${config.storeMemories !== false}`);
          console.log(`  Store transcripts: ${config.storeTranscripts !== false}`);
          console.log(`  Store audio:       ${config.storeAudio === true}`);
          console.log(`  Memory path:       ${config.memoryPath || "memory/omi"}`);
          console.log(`  Notify agent:      ${config.notifyAgent === true}`);
        });

      cmd
        .command("urls")
        .description("Show webhook URLs to configure in Omi app")
        .action(() => {
          const gatewayPort = api.config?.gateway?.port || 18789;
          const baseUrl = `http://localhost:${gatewayPort}`;
          
          console.log("🔗 OmiMesh — Configure these URLs in Omi Developer Settings:");
          console.log("");
          console.log("Conversation Events (Memory):");
          console.log(`  ${baseUrl}/omimesh/memory`);
          console.log("");
          console.log("Real-time Transcript:");
          console.log(`  ${baseUrl}/omimesh/transcript`);
          console.log("");
          console.log("Audio Bytes:");
          console.log(`  ${baseUrl}/omimesh/audio`);
          console.log("");
          console.log("Day Summary:");
          console.log(`  ${baseUrl}/omimesh/memory  (same as Conversation Events)`);
          console.log("");
          console.log("Note: Use ngrok or similar for external access:");
          console.log(`  ngrok http ${gatewayPort}`);
        });
    },
    { commands: ["omimesh"] }
  );

  logger.info("[omimesh] 🔗 Plugin ready — endpoints: /omimesh/*");
}
